package com.paydevice.smartpos.demo;

import android.Manifest;
import android.app.Activity;
import android.content.pm.PackageManager;
import android.media.AudioManager;
import android.media.ImageReader;
import android.media.SoundPool;
import android.os.Build;
import android.os.Bundle;
import android.util.Log;

import android.view.View;
import android.view.View.OnClickListener;
import android.widget.FrameLayout;
import android.widget.Button;
import android.widget.TextView;

import java.lang.NullPointerException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;

/* Import ZBar Class files */
import net.sourceforge.zbar.ImageScanner;
import net.sourceforge.zbar.Image;
import net.sourceforge.zbar.Symbol;
import net.sourceforge.zbar.SymbolSet;
import net.sourceforge.zbar.Config;

public class CameraActivity extends Activity {

    private static final String TAG = "CameraActivity";
	private static final int PERMISSION_REQUEST_CODE = 100;
	private static final int FILTER_FRAME_COUNT = 3;

	private MyToast mToast;
    TextView scanText;
    Button flashButton;

    private SoundPool mSoundPool;
    private int mBeepSoundId;

	private CameraPreview mPreview;
    private ImageScanner scanner;

    static {
        System.loadLibrary("iconv");//for qrcode
    }

    private SymbolSet mLastSymbolSet;
    private int mRepeatCount;

    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.camera);
        setTitle(R.string.camera);

		mToast = new MyToast(this);

        /* Instance barcode scanner */
        scanner = new ImageScanner();
        scanner.setConfig(0, Config.X_DENSITY, 2);
        scanner.setConfig(0, Config.Y_DENSITY, 2);
		/* recommend only enable code type you need */
		//scanner.setConfig(Symbol.QRCODE, Config.ENABLE, 1);
		//scanner.setConfig(Symbol.EAN8, Config.ENABLE, 1);
		//scanner.setConfig(Symbol.EAN13, Config.ENABLE, 1);
		//scanner.setConfig(Symbol.CODE128, Config.ENABLE, 1);
		//scanner.setConfig(Symbol.CODE39, Config.ENABLE, 1);
		//scanner.setConfig(Symbol.CODE93, Config.ENABLE, 1);
		//scanner.setConfig(Symbol.UPCA, Config.ENABLE, 0);
		//scanner.setConfig(Symbol.UPCE, Config.ENABLE, 0);

        scanText = findViewById(R.id.scanText);

        flashButton = findViewById(R.id.FlashButton);
        flashButton.setOnClickListener(new OnClickListener() {
            public void onClick(View v) {
                if (flashButton.getText().equals("Flash On")) {
					mPreview.flashOn();
                    flashButton.setText("Flash Off");
                } else {
					mPreview.flashOff();
                    flashButton.setText("Flash On");
                }
            }
        });
    }

    public void onResume() {
        super.onResume();

        initBeep();
		String[] permissions = {Manifest.permission.CAMERA};
		if (Build.VERSION.SDK_INT >= 23
				&& checkSelfPermission(permissions[0]) != PackageManager.PERMISSION_GRANTED) {
			requestPermissions(permissions, PERMISSION_REQUEST_CODE);
		} else {
			openCamera();
		}
        mRepeatCount = 0;
    }

    public void onPause() {
        super.onPause();
		mToast.cancel();
        Log.d(TAG, "release camera");
        releaseBeep();
    }

	@Override
	public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
		if (requestCode == PERMISSION_REQUEST_CODE) {
			if (grantResults.length > 0 && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
				Log.d(TAG,"camera permissions granted");
				openCamera();
			} else {
				Log.d(TAG,"camera permissions denied");
				mToast.showToast(R.string.camera_denied);
			}
		}
	}

	private void openCamera() {
		try {
			mPreview = new CameraPreview(this, new ImageReader.OnImageAvailableListener() {
				@Override
				public void onImageAvailable(ImageReader reader) {
					try {
						android.media.Image image = reader.acquireLatestImage();
						ByteBuffer buffer = image.getPlanes()[0].getBuffer();
						byte[] data = new byte[buffer.remaining()];
						buffer.get(data);
 		    	    	Image barcode = new Image(image.getWidth(), image.getHeight(), "Y800");
 		    	    	barcode.setData(data);

						//NOTE:maybe move decode to handler thread to avoid main thread block
 		    	    	int result = scanner.scanImage(barcode);
 		    	    	if (result != 0) {
                            /* Sometimes zbar decode a non-barcode frame as a barcode frame, so we need filter these wrong result.
							 * we think there three same decode result then the 3rd result is vaild
							 */
 		    	    	    SymbolSet syms = scanner.getResults();
                             if (!syms.equals(mLastSymbolSet) && mRepeatCount == FILTER_FRAME_COUNT-1) {
                                 playBeep();
                                 scanText.setText("");
                                 for (Symbol sym : syms) {
                                     byte[] symBytes = sym.getDataBytes();
                                     scanText.append("Type: " + typeToString(sym.getType()) + "\nCode: " + new String(symBytes, StandardCharsets.UTF_8));
									 int[] location = sym.getLocationPoint(0);
                                     Log.d(TAG, "Type: " + typeToString(sym.getType()) + " Code: " + new String(symBytes, StandardCharsets.UTF_8) + " Quality:" + sym.getQuality()
									 + " Orientation:" + sym.getOrientation() + " xy:" + location[0] + " " + location[1]);
                                 }
                                 mLastSymbolSet = scanner.getResults();
                                 mRepeatCount = 0;
                             } else {
                                 for (Symbol sym : syms) {
                                     byte[] symBytes = sym.getDataBytes();
                                     Log.d(TAG, "filter result,Type: " + typeToString(sym.getType()) + " Code: " + new String(symBytes, StandardCharsets.UTF_8));
                                 }
                                 mRepeatCount++;
                             }
                        }
						image.close();
					} catch (NullPointerException ignored) {
					}
				}
			});
			FrameLayout preview = findViewById(R.id.cameraPreview);
			preview.addView(mPreview);
			flashButton.setText("Flash On");
			scanText.setText("Scanning...");
			if (!mPreview.cameraExist()) {
				mToast.showToast(R.string.camera_error);
			} else {
				mToast.showToast(R.string.camera_tips);
			}
		} catch (RuntimeException e) {
			e.printStackTrace();
			Log.d(TAG, "open camera failed");
			mToast.showToast(R.string.camera_error);
			finish();
		}
	}

    private String typeToString(int type) {
        String str;
        switch (type) {
            case Symbol.EAN8:
                str = "EAN8";
                break;
            case Symbol.UPCE:
                str = "UPCE";
                break;
            case Symbol.ISBN10:
                str = "ISBN10";
                break;
            case Symbol.UPCA:
                str = "UPCA";
                break;
            case Symbol.EAN13:
                str = "EAN13";
                break;
            case Symbol.ISBN13:
                str = "ISBN13";
                break;
            case Symbol.I25:
                str = "I25";
                break;
            case Symbol.DATABAR:
                str = "DATABAR";
                break;
            case Symbol.DATABAR_EXP:
                str = "DATABAR_EXP";
                break;
            case Symbol.CODABAR:
                str = "CODABAR";
                break;
            case Symbol.CODE39:
                str = "CODE39";
                break;
            case Symbol.PDF417:
                str = "PDF417";
                break;
            case Symbol.QRCODE:
                str = "QRCODE";
                break;
            case Symbol.CODE93:
                str = "CODE93";
                break;
            case Symbol.CODE128:
                str = "CODE128";
                break;
            default:
                str = "None";
                break;
        }
        return str;
    }

    private void initBeep() {
        if (mSoundPool == null) {
            if (Build.VERSION.SDK_INT >= 21) {
                mSoundPool = new SoundPool.Builder().setMaxStreams(7).build();
            } else {
                mSoundPool = new SoundPool(7, AudioManager.STREAM_MUSIC, 1);
            }
        }
        mBeepSoundId = mSoundPool.load(this, R.raw.beep, 1);
    }

    private void releaseBeep() {
        if (mSoundPool != null) {
            mSoundPool.release();
            mSoundPool = null;
        }
    }

    private void playBeep() {
        if (mSoundPool != null) {
            mSoundPool.play(mBeepSoundId, 1.0f, 1.0f, 0, 0, 1.0f);
        }
    }
}
