package com.paydevice.smartpos.demo;

import android.support.v7.app.AppCompatActivity;
import android.os.Bundle;
import android.util.Log;
import android.view.KeyEvent;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.TextView;
import android.widget.Button;

import com.paydevice.smartpos.sdk.SmartPosException;
import com.paydevice.smartpos.sdk.magneticcard.MagneticCardManager;

import java.util.Timer;
import java.util.TimerTask;


public class McrActivity extends AppCompatActivity {

    private static final String TAG = "McrActivity";

	private MyToast mToast;
    private TextView resultText;

    private MagneticCardManager mMagneticCardManager;
    private int mReaderType = MagneticCardManager.MCR_TYPE_UART;

    //for UART MSR
    private static final int TASK_RUNNING = 10;
    private static final int TASK_CANCEL = 11;
    private Timer readTimer;
    private TimerTask readTask;
    private int taskStatus = TASK_CANCEL;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_mcr);
        setTitle(R.string.mcr);
        resultText = (TextView) this.findViewById(R.id.result);

		mToast = new MyToast(this);
        Button cleanBtn = (Button) this.findViewById(R.id.clean_btn);
        cleanBtn.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View arg) {
                resultText.setText("");
            }
        });

		mMagneticCardManager = new MagneticCardManager();
        try {
            mMagneticCardManager.init();
        } catch (SmartPosException e) {
            Log.d(TAG, "err code:" + e.getErrorCode());
            if (e.getErrorCode() == MagneticCardManager.MCR_ERR_INIT) {
                resultText.setText(R.string.mcr_init_err);
            }
        }
    }

    @Override
    protected void onResume() {
        super.onResume();
        mToast.showToast(R.string.mcr_swipe);

        //poll tracks data for UART reader
        if (mReaderType == MagneticCardManager.MCR_TYPE_UART) {
            startRead();
        }
    }

    @Override
    public void onPause() {
        super.onPause();
        if (mReaderType == MagneticCardManager.MCR_TYPE_UART) {
            stopRead();
        }
		mToast.cancel();
		resultText.setText("");
    }

    @Override
    protected void onDestroy() {
        if (mMagneticCardManager != null) {
            mMagneticCardManager.deinit();
        }
        super.onDestroy();
    }

    /**
    * @brief key event only work for SPI interface MSR, for UART interface MSR could only poll tracks data
    *
    * @param keyCode
    * @param event
    *
    * @return 
    */
    @Override
    public boolean onKeyDown(int keyCode, KeyEvent event) {
        //receive swipe card event and then read tracks data
        if (keyCode == MagneticCardManager.MCR_SWIPE_CARD_PRESENT) {
            Log.d(TAG, "swipe card:" + keyCode);
            resultText.setText("");
            showTracksData();
            return true;
        }
        return super.onKeyDown(keyCode, event);
    }

    private void stopRead() {
        taskStatus = TASK_CANCEL;
        if (readTask != null) {
			Log.d(TAG, "---clean read task---");
            readTask.cancel();
            readTask = null;
        }
        if (readTimer != null) {
            readTimer.cancel();
            readTimer.purge();
            readTimer = null;
        }
    }
    
    private void startRead() {
        if (readTimer == null) {
            readTimer = new Timer();
        }
        if (readTask != null) {
            readTask.cancel();
            readTask = null;
        }

        readTask = new TimerTask() {
            public void run() {
                int status = getTaskStatus();
                switch (status) {
                    case TASK_RUNNING:
                        runOnUiThread(() -> {
                            showTracksData();
                        });
                        break;

                    case TASK_CANCEL:
                        stopRead();
                        break;
                }
            }
        };
        taskStatus = TASK_RUNNING;
        readTimer.schedule(readTask, 0, 1000);
    }

    public synchronized int getTaskStatus() {
        return taskStatus;
    }

    private void showTracksData() {
        try {
            if (mMagneticCardManager.query() == 0) {
                //get 3 tracks data
                byte[] track1 = mMagneticCardManager.getTrack(1);
                byte[] track2 = mMagneticCardManager.getTrack(2);
                byte[] track3 = mMagneticCardManager.getTrack(3);
                //get track 1 error code
                if (track1.length == 1) {
                    if (track1[0] == MagneticCardManager.MCR_ERR_CODE_BLANK_TRACK) {
                        Log.d(TAG, "track 1 was blank");
                        //break;
                    }
                    Log.d(TAG, "track 1 err code: 0x" + Integer.toHexString(track1[0] & 0xff));
                }
                //get track 2 error code
                if (track2.length == 1) {
                    if (track2[0] == MagneticCardManager.MCR_ERR_CODE_BLANK_TRACK) {
                        Log.d(TAG, "track 2 was blank");
                        //break;
                    }
                    Log.d(TAG, "track 2  err code: 0x" + Integer.toHexString(track2[0] & 0xff));
                }
                //get track 3 error code
                if (track3.length == 1) {
                    if (track3[0] == MagneticCardManager.MCR_ERR_CODE_BLANK_TRACK) {
                        Log.d(TAG, "track 3 was blank");
                        //break;
                    }
                    Log.d(TAG, "track 3  err code: 0x" + Integer.toHexString(track3[0] & 0xff));
                }

                String out = "track 1(" + track1.length + "):\n"
                    + new String(track1) + "\n"
                    + "track 2(" + track2.length + "):\n"
                    + new String(track2) + "\n"
                    + "track 3(" + track3.length + "):\n"
                    + new String(track3) + "\n";
                resultText.setText(out);
            } else {
                //sometimes the tracks data maybe incomplete, try again 
                mToast.showToast(R.string.mcr_read_err);
            }
        } catch (SmartPosException e) {
            Log.d(TAG, "err code:" + e.getErrorCode());
        }
    }
}

