/**
 * @file PosSalesSlip.java
 * @brief a printing template
 * @author Hansen.Z
 * @version 1.0
 * @date 2015-04-24
 */
package com.paydevice.smartpos.demo.printer;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Typeface;
import android.os.Build;
import android.support.annotation.RequiresApi;
import android.text.Layout;
import android.text.StaticLayout;
import android.text.TextDirectionHeuristics;
import android.text.TextPaint;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.Log;
import android.view.View;

import com.paydevice.smartpos.demo.R;
import com.paydevice.smartpos.demo.Utils;
import com.paydevice.smartpos.sdk.SmartPosException;
import com.paydevice.smartpos.sdk.printer.PrinterManager;
import com.paydevice.smartpos.sdk.ImageUtils;

import java.util.Locale;


/**
* @brief a printing template
*/
public class PosSalesSlip {

	private static String TAG = "PosSalesSlip";

	private String dividingLine;
	/** Printer object */
	private PrinterManager mPrinterManager;
	/** Application context */
	private Context mContext;

	private String merchantName;
	private String merchantNo;
	private String operatorNo;
	private String terminalNo;
	private String issuer;
	private String acquirer;
	private String cardNo;
	private String txnType;
	private String batchNo;
	private String voucherNo;
	private String authNo;
	private String expDate;
	private String refNo;
	private String date;
	private String amount;
	private String total;
	private String reference;
	private String phone;

    private int mTemplateId;

	public PosSalesSlip(Context context, PrinterManager printer) {
		mContext = context;
		mPrinterManager = printer;
	}

	/**
	* @brief initialize printer and check the paper
	*
	* @return 0: success other: error code
	*/
	public int prepare() {
		try {
			mPrinterManager.connect();
			//if (mPrinterManager.getPrinterType() == PrinterManager.PRINTER_TYPE_SERIAL) {
			//	if (mPrinterManager.cmdGetPrinterModel() == PrinterManager.PRINTER_MODEL_PRN2103) {
			//		Log.d(TAG,"model:PRN 2103");	
			//	} else {
			//		Log.d(TAG,"model:UNKNOWN");	
			//	}
			//}
			mPrinterManager.checkPaper();
        } catch (SmartPosException e) {
            return e.getErrorCode();
        }
		return 0;
	}

	/**
	* @brief deinitialize printer
	*
	* @return
	*/
	public int destroy() {
		/* NOTE: 1.built-in serialport printer have a power control IO, such as models FH070H-A/FH100H-A/FH100-A3-D.
		   2.Invoke disconnect(true) to turn off then printer power if your device used battery supply.
		   Of course you can choose keep the power of the printer.
		   3.Because of there are serval seconds dealy before turn off, so you should not invoke disconnect(true) in UI thread otherwise maybe cause ANR.
		 */
		try {
			if (mPrinterManager.getPrinterType() == PrinterManager.PRINTER_TYPE_SERIAL) {
				mPrinterManager.disconnect(true);//disconnect and turn off printer power.
			} else {
				mPrinterManager.disconnect();
			}
		} catch (SmartPosException e) {
			return e.getErrorCode();
		}
		Log.d(TAG,"disconnect printer");
		return 0;
	}

	/**
	* @brief print template
	*
	* @return 
	*/
	synchronized public void print() throws SmartPosException {
		//NOTE: For different content(even different paper quality) need adjust heating parameter to obtain better print quality 
		if (mPrinterManager.isBuiltInSlow()) {
			mPrinterManager.cmdSetHeatingParam(7, 140, 2);
		} else {
			mPrinterManager.cmdSetHeatingParam(15, 100, 10);//fast speed
		}

		printUnsupportLanguage();
		languageTest();
		printLogo();
		printTemplate();
		printQRcode();
		printBarcode();
		printModeTest();
		printTableTest();
		//mPrinterManager.cmdPrintTest();

		mPrinterManager.cmdLineFeed(3);
		if (mPrinterManager.getPrinterType() == PrinterManager.PRINTER_TYPE_USB) {
			mPrinterManager.cmdCutPaper(PrinterManager.FULL_CUT);
		}
	}

	//debug
	private void languageTest() throws SmartPosException {
		mPrinterManager.cmdSetAlignMode(PrinterManager.ALIGN_LEFT);
		//Only usb printer support these code page
		//if (mPrinterManager.getPrinterType() == PrinterManager.PRINTER_TYPE_USB) {
		//	mPrinterManager.setStringEncoding("Shift_JIS");
		//	mPrinterManager.cmdSetPrinterLanguage(PrinterManager.CODE_PAGE_SHIFT_JIS);
		//	printLine("Japanese","こんにちは世界");

		//	mPrinterManager.setStringEncoding("EUC-KR");
		//	mPrinterManager.cmdSetPrinterLanguage(PrinterManager.CODE_PAGE_EUC_KR);
		//	printLine("Korean","안녕 세상");
		//}

		mPrinterManager.setStringEncoding("CP437");
		mPrinterManager.cmdSetPrinterLanguage(PrinterManager.CODE_PAGE_CP437);
		printLine("English","Hello World");
		printLine("German","Hallo Welt");
		printLine("French","Bonjour le monde");

		mPrinterManager.setStringEncoding("CP1251");
		mPrinterManager.cmdSetPrinterLanguage(PrinterManager.CODE_PAGE_CP1251);
		printLine("Russian","Привет мир");

		mPrinterManager.setStringEncoding("CP775");
		mPrinterManager.cmdSetPrinterLanguage(PrinterManager.CODE_PAGE_CP775);
		printLine("Latvian","Sveiki Pasaule");

		mPrinterManager.setStringEncoding("ISO-8859-2");
		mPrinterManager.cmdSetPrinterLanguage(PrinterManager.CODE_PAGE_ISO_8859_2);
		printLine("Romanian","Bună ziua lumea");

		mPrinterManager.setStringEncoding("GB18030");
		mPrinterManager.cmdSetPrinterLanguage(PrinterManager.CODE_PAGE_GB18030);
		printLine("Chinese","你好世界");

		//mPrinterManager.setStringEncoding("BIG5");
		//mPrinterManager.cmdSetPrinterLanguage(PrinterManager.CODE_PAGE_BIG5);
		//printLine("Chinese","繁體字",false,true));
	}

    /**
     * @brief Print text and feed one line by printer built-in codepage
     *
     * @param data print text
     *
     * @return 
     */
	private void printLine(String data) throws SmartPosException {
		mPrinterManager.sendData(data);
		mPrinterManager.cmdLineFeed();
	}

	/**
	* @brief Print one line text table with 2 columns by printer built-in codepage, left text align left and right text align right
	*
	* @param leftText left text
	* @param rightText right text
	*
	* @return 
	*/
	private void printLine(String leftText, String rightText) throws SmartPosException {
		String[] arrStr = new String[] {leftText, rightText};
		int[] arrWidth = new int[] {1, 2};//ratio 1:2
		int[] arrAlign = new int[] {PrinterManager.ALIGN_LEFT, PrinterManager.ALIGN_RIGHT};
		printLine(arrStr, arrWidth, arrAlign);
	}

    /**
     * @brief Print one line text table with multi columns by printer built-in codepage, implemented with blank
     *
     * @param arrStr items array
     * @param arrWidth items width array
     * @param arrAlign items align mode array
     *
     * @return 
     */
    private void printLine(String[] arrStr, int[] arrWidth, int[] arrAlign) throws SmartPosException {
        String data = ""; 
		final int totalDots = mPrinterManager.getDotsPerLine();
        int totalRatio = 0;
        for(int ratio : arrWidth) {
            totalRatio += ratio;
        }
        for(int i=0; i<arrStr.length; i++) {
            int width = Math.round((totalDots/12) * (arrWidth[i] / (float)totalRatio));
            String item = (getCount(arrStr[i]) > width) ? (arrStr[i].substring(0, width)) : (arrStr[i]);//cut string if too long
            if(arrAlign[i] == PrinterManager.ALIGN_LEFT){
                data += (item + getBlank(width - getCount(item)));
            } else if (arrAlign[i] == PrinterManager.ALIGN_MIDDLE) {
                String leftBlank = getBlank((width - getCount(item)) / 2);
                String rightBlank = leftBlank;
                int rightLen = width - getCount(item) - leftBlank.length() * 2;
                if(rightLen > 0) { 
                    rightBlank += getBlank(rightLen);
                }
                data += leftBlank + item + rightBlank;
            } else if(arrAlign[i] == PrinterManager.ALIGN_RIGHT){
                data += (getBlank(width - getCount(item)) + item);
            }
        }
		mPrinterManager.sendData(data);
		mPrinterManager.cmdLineFeed();
	}

	/**
	* @brief Print one line text by bitmap print.
	*
	* @param data
	*
	* @return 
	*/
	private void printLineByBitmap(String data) throws SmartPosException {
		String[] arrStr = new String[] {data};
		int[] arrWidth = new int[] {100};//ratio 100%
		int[] arrAlign = new int[] {PrinterManager.ALIGN_LEFT};
		printLineByBitmap(arrStr, arrWidth, arrAlign);
	}

	/**
	* @brief Print one line text table with 2 columns by bitmap print, left text align left and right text align right
	*
	* @param leftText left text
	* @param rightText right text
	*
	* @return 
	*/
	private void printLineByBitmap(String leftText, String rightText) throws SmartPosException {
		String[] arrStr = new String[] {leftText, rightText};
		int[] arrWidth = new int[] {1, 2};//ratio 1:2
		int[] arrAlign = new int[] {PrinterManager.ALIGN_LEFT, PrinterManager.ALIGN_RIGHT};
		printLineByBitmap(arrStr, arrWidth, arrAlign);
	}

	/**
	* @brief Print one line text table with multi columns by bitmap print. 
	*
    * @param arrStr items array
    * @param arrWidth items width array
    * @param arrAlign items align mode array
	*
	* @return 
	*/
	private void printLineByBitmap(String[] arrStr, int[] arrWidth, int[] arrAlign) throws SmartPosException {
		final float size = 24f;//default text size is 24px(24dots)
		final float lineSpacingAdd = 4f;
		final int lineWidth = mPrinterManager.getDotsPerLine();
		if (arrStr.length > 3) {
			Log.w(TAG,"string too long, maybe the string cannot printed completely");
		}
		//setup text paint
		TextPaint textPaint = new TextPaint();
		textPaint.setColor(Color.BLACK); 
		textPaint.setStyle(Paint.Style.FILL_AND_STROKE);
		textPaint.setStrokeCap(Paint.Cap.SQUARE);
		textPaint.setStrokeWidth(0);
		textPaint.setTextSize(size);
		//android system font include most languages of world, but you can also used Typefase to load external .ttf/.otf font.
		Typeface font = Typeface.create(Typeface.SANS_SERIF, Typeface.NORMAL);
		textPaint.setTypeface(font);
		//metrics font
		Paint.FontMetrics fontMetrics = textPaint.getFontMetrics();
		final float yPos = fontMetrics.descent-fontMetrics.ascent-size;
		final int lineHeight = Math.round(fontMetrics.descent-fontMetrics.ascent+yPos+lineSpacingAdd);

		int totalRatio = 0;
		for(int ratio : arrWidth) {
			totalRatio += ratio;
		}
		//use a bitmap to load one line string, bitmap width is always lineWidth(384px for 58mm).
		//The part that exceeds the width will not be print
		Bitmap bmp = Bitmap.createBitmap(lineWidth, lineHeight, Bitmap.Config.ARGB_8888);
		Canvas canvas = new Canvas(bmp);
		canvas.drawColor(Color.WHITE);

		int previousStrGraphWidth = 0;
		//draw each text part to bitmap
		for(int i=0; i<arrStr.length; i++) {
			int strGraphWidth = arrWidth[i]*lineWidth/totalRatio;
			if (strGraphWidth < 12) {
				Log.e(TAG,"width too small,check the ratio");
				continue;
			}
			StaticLayout layout = null;
			//android 7.1+
			if (Build.VERSION.SDK_INT > 22) {
				StaticLayout.Builder builder = StaticLayout.Builder.obtain(arrStr[i], 0, arrStr[i].length(), textPaint, strGraphWidth)
					.setTextDirection(TextDirectionHeuristics.LTR)//Always decides that the direction is left to right.
					.setLineSpacing(lineSpacingAdd, 1.0f)//line spacing mult 1.0
					.setIncludePad(false);
				//set align mode for each string
				switch(arrAlign[i]) {
					case PrinterManager.ALIGN_LEFT:
						builder.setAlignment(Layout.Alignment.ALIGN_NORMAL);
						break;
					case PrinterManager.ALIGN_MIDDLE:
						builder.setAlignment(Layout.Alignment.ALIGN_CENTER);
						break;
					case PrinterManager.ALIGN_RIGHT:
						builder.setAlignment(Layout.Alignment.ALIGN_OPPOSITE);
						break;
				}
				layout = builder.build();
			} else {
				switch(arrAlign[i]) {
					case PrinterManager.ALIGN_LEFT:
						layout = new StaticLayout(arrStr[i], textPaint, strGraphWidth, Layout.Alignment.ALIGN_NORMAL, 1.0f, 0.0f, false);
						break;
					case PrinterManager.ALIGN_MIDDLE:
						layout = new StaticLayout(arrStr[i], textPaint, strGraphWidth, Layout.Alignment.ALIGN_CENTER, 1.0f, 0.0f, false);
						break;
					case PrinterManager.ALIGN_RIGHT:
						layout = new StaticLayout(arrStr[i], textPaint, strGraphWidth, Layout.Alignment.ALIGN_OPPOSITE, 1.0f, 0.0f, false);
						break;
				}
			}
			//Log.d(TAG,"lines:"+layout.getLineCount()+" lineHeight:"+lineHeight+" strWidth:"+strGraphWidth);
			//Log.d(TAG,"top:"+fontMetrics.top+" leading:"+fontMetrics.leading+" ascent:"+fontMetrics.ascent
			//		+" yPos:"+yPos+" descent:"+fontMetrics.descent+" bottom:"+fontMetrics.bottom);
			canvas.translate(previousStrGraphWidth, yPos);
			layout.draw(canvas);
			previousStrGraphWidth += strGraphWidth;
		}
		mPrinterManager.cmdBitmapPrint(bmp, 0, 0);
		//try used cmdSetLineSpacing and cmdLineFeed(n) if you want setup line spacing for printLineByBitmap

		bmp.recycle();
		bmp = null;
	}

	private void printMulitLineByBitmap(String text) throws SmartPosException {
		final float lineSpacingAdd = 4f;
		final int bmpWidth = mPrinterManager.getDotsPerLine();
		TextPaint textPaint = new TextPaint();
		Typeface font = Typeface.create(Typeface.SANS_SERIF, Typeface.NORMAL);
		textPaint.setColor(Color.BLACK); 
		textPaint.setStyle(Paint.Style.FILL_AND_STROKE);
		textPaint.setStrokeCap(Paint.Cap.SQUARE);
		textPaint.setStrokeWidth(0);
		textPaint.setTextSize(24); 
		textPaint.setTypeface(font);
		Paint.FontMetrics fontMetrics = textPaint.getFontMetrics();
		final float yPos = fontMetrics.descent-fontMetrics.ascent-24;

		mPrinterManager.checkPaper();
		String[] lines = text.split("\\n");
		for (String str: lines) {
			StaticLayout.Builder builder = StaticLayout.Builder.obtain(str, 0, str.length(), textPaint, bmpWidth)
				.setLineSpacing(lineSpacingAdd, 1.0f)//line spacing mult 1.0
				.setIncludePad(false);
			StaticLayout layout = builder.build();
			final int bmpHeight = Math.round(layout.getLineCount()*(fontMetrics.descent-fontMetrics.ascent+lineSpacingAdd)+yPos);
			Bitmap bmp = Bitmap.createBitmap(bmpWidth, bmpHeight, Bitmap.Config.ARGB_8888);
			Canvas canvas = new Canvas(bmp);
			canvas.drawColor(Color.WHITE);
			canvas.translate(0, yPos);
			layout.draw(canvas);
			Log.d(TAG,"lines:"+layout.getLineCount()+" bmpHeight:"+bmpHeight);
			Log.d(TAG,"top:"+fontMetrics.top+" leading:"+fontMetrics.leading+" ascent:"+fontMetrics.ascent
					+" yPos:"+yPos+" descent:"+fontMetrics.descent+" bottom:"+fontMetrics.bottom);
			mPrinterManager.cmdBitmapPrint(bmp, 0, 0);
			bmp.recycle();
			bmp = null;
		}
	}

	//test print world languages by bitmap print
	private void printUnsupportLanguage() throws SmartPosException {
		//mPrinterManager.cmdSetHeatingParam(7, 100, 10);//slow speed, more clearly

		printLineByBitmap("English", "Hello World");
		printLineByBitmap("Arabic", "مرحبا بالعالم");
		printLineByBitmap("Malayalam", "ഹലോ വേൾഡ്");
		printLineByBitmap("Hindi", "नमस्ते दुनिया");
		printLineByBitmap("Turkish", "Selam Dünya");
		printLineByBitmap("Hebrew", "שלום עולם");
		printLineByBitmap("Japanese", "こんにちは世界");
		printLineByBitmap("Korean", "안녕 세상");
		printLineByBitmap("Thai", "สวัสดีชาวโลก");
		printLineByBitmap("Vietnamese", "Chào thế giới");
		printLineByBitmap("Burmese", "မင်္ဂလာပါကမ္ဘာလောက");

		mPrinterManager.cmdLineFeed(1);
	}

	//test table command
	private void printTableTest() {
		try {
			//restore default font style and align mode
			mPrinterManager.cmdSetPrintMode(PrinterManager.FONT_DEFAULT);
			mPrinterManager.cmdSetAlignMode(PrinterManager.ALIGN_LEFT);
			//table have 3 columns, offsets: 2,9,19 unit is 12dots
			byte[] offsets = new byte[] {(byte)2, (byte)9, (byte)19};

			mPrinterManager.cmdSetTable(offsets);//set table
			mPrinterManager.cmdJumpTab();//jump first offset, just comment out if want print item1 as start of a line.
			mPrinterManager.sendData("itme1");
			mPrinterManager.cmdJumpTab();//jump second offset
			mPrinterManager.sendData("itme2");
			mPrinterManager.cmdJumpTab();//jump third offset
			mPrinterManager.sendData("itme3");
			mPrinterManager.cmdLineFeed();

			mPrinterManager.cmdJumpTab();
			mPrinterManager.sendData("apple");
			mPrinterManager.cmdJumpTab();
			mPrinterManager.sendData("banana");
			mPrinterManager.cmdJumpTab();
			mPrinterManager.sendData("cherry");
			mPrinterManager.cmdLineFeed();
			mPrinterManager.cmdUnSetTable();//unset table

			mPrinterManager.sendData("12345678901234567890123456789012");//reference line, used to check all offset
			mPrinterManager.cmdLineFeed();

			//test multi columns table
			String[] dataArr = new String[] {"Pizza","x123","123.23"};
			int[] widthArr = new int[] {6,3,3};
			int[] alignArr = new int[] {0,2,2};
			printLine(dataArr, widthArr, alignArr);
		} catch (SmartPosException e) {
		}
	}

	private void printModeTest() {
		try {
			mPrinterManager.cmdSetAlignMode(PrinterManager.ALIGN_LEFT);
			mPrinterManager.cmdSetPrintMode(PrinterManager.FONT_UPSIDE_DOWN|PrinterManager.FONT_UNDERLINE);
			mPrinterManager.cmdSetUnderlineHeight(2);
			mPrinterManager.sendData("upside down and underline");
			mPrinterManager.cmdLineFeed();

			mPrinterManager.cmdSetPrintMode(PrinterManager.FONT_ROTATE|PrinterManager.FONT_EMPHASIZED);
			mPrinterManager.sendData("rotate");
			mPrinterManager.cmdLineFeed();

			mPrinterManager.cmdSetPrintMode(PrinterManager.FONT_DEFAULT);
			mPrinterManager.cmdSetFontScaleSize(1,1);
			mPrinterManager.sendData("AaBbCc");
			mPrinterManager.cmdLineFeed();

			//adjust for inverse
			if (mPrinterManager.isBuiltInSlow()) {
				mPrinterManager.cmdSetHeatingParam(7, 60, 2);
			} else {
				mPrinterManager.cmdSetHeatingParam(15, 60, 10);
			}
			mPrinterManager.cmdSetPrintMode(PrinterManager.FONT_INVERSE|PrinterManager.FONT_SMALL);
			mPrinterManager.sendData("inverse and small");
			mPrinterManager.cmdLineFeed();
			//restore heating parameter
			if (mPrinterManager.isBuiltInSlow()) {
				mPrinterManager.cmdSetHeatingParam(7, 140, 2);
			} else {
				mPrinterManager.cmdSetHeatingParam(15, 100, 10);
			}
		} catch (SmartPosException e) {
		}
	}

	private void printDevidingLine() throws SmartPosException {
		final int totalDots = mPrinterManager.getDotsPerLine();
		int[] pLineStartPosition_58 = {0,192};
		int[] pLineEndPosition_58 = {192,383};
		int[] pLineStartPosition_80 = {0,200,400};
		int[] pLineEndPosition_80 = {200,400,575};
		int height=5;
		while(height-- > 0) {
			try {
				if (totalDots == 384)
					mPrinterManager.cmdPrintMultipleLines(pLineStartPosition_58.length, pLineStartPosition_58, pLineEndPosition_58);
				else
					mPrinterManager.cmdPrintMultipleLines(pLineStartPosition_80.length, pLineStartPosition_80, pLineEndPosition_80);
			} catch (SmartPosException e) {
			}
		}
		mPrinterManager.cmdLineFeed(1);
	}

	private void printTemplate() {
		try {
			//set encoding and codepage
			Locale locale = mContext.getResources().getConfiguration().locale;
			String language = locale.getLanguage();
			if (language.endsWith("zh")) {
				mPrinterManager.cmdSetPrinterLanguage(PrinterManager.CODE_PAGE_GB18030);
				mPrinterManager.setStringEncoding("GB18030");
			} else {
				mPrinterManager.cmdSetPrinterLanguage(PrinterManager.CODE_PAGE_CP437);
				mPrinterManager.setStringEncoding("CP437");
			}

			//set font style for title
			mPrinterManager.cmdSetPrintMode(PrinterManager.FONT_DOUBLE_HEIGHT
					|PrinterManager.FONT_DOUBLE_WIDTH
					|PrinterManager.FONT_EMPHASIZED);
			mPrinterManager.cmdSetAlignMode(PrinterManager.ALIGN_MIDDLE);
			//title
			printLine(mContext.getString(R.string.tag_title));
			mPrinterManager.cmdLineFeed();
			//restore default font style and align mode
			mPrinterManager.cmdSetPrintMode(PrinterManager.FONT_DEFAULT);
			mPrinterManager.cmdSetAlignMode(PrinterManager.ALIGN_LEFT);
			//header
			printLine(mContext.getString(R.string.tag_page_header));
			printDevidingLine();
			//content
			printLine(mContext.getString(R.string.tag_merchant_name), merchantName);
			printLine(mContext.getString(R.string.tag_merchant_no), merchantNo);
			printLine(mContext.getString(R.string.tag_terminal_no), terminalNo);
			printLine(mContext.getString(R.string.tag_date), date);
			printLine(mContext.getString(R.string.tag_issuer), issuer);
			printLine(mContext.getString(R.string.tag_card_no), cardNo);
			printLine(mContext.getString(R.string.tag_exp_date), expDate);
			printLine(mContext.getString(R.string.tag_txn_type), txnType);
			printLine(mContext.getString(R.string.tag_batch_no), batchNo);
			printLine(mContext.getString(R.string.tag_voucher_no), voucherNo);
			printLine(mContext.getString(R.string.tag_ref_no), refNo);
			printLine(mContext.getString(R.string.tag_auth_no), authNo);
			printLine(mContext.getString(R.string.tag_operator_no), operatorNo);
			printLine(mContext.getString(R.string.tag_amount), amount);
			printLine(mContext.getString(R.string.tag_reference));
			printDevidingLine();
			//footer
			printLine(mContext.getString(R.string.tag_page_footer));
		} catch (SmartPosException e) {
		}
	}

	private void printLogoFromNVRAM() {
		try {
			//just print first bitmap(index=1,2) from NVRAM
			mPrinterManager.cmdSetAlignMode(PrinterManager.ALIGN_MIDDLE);
			mPrinterManager.cmdPrintBitmapFromNVRAM(1, PrinterManager.BITMAP_ZOOM_NONE);//logo.bmp
			mPrinterManager.cmdPrintBitmapFromNVRAM(2, PrinterManager.BITMAP_ZOOM_NONE);//logo_large.bmp
			mPrinterManager.cmdLineFeed();
		} catch (SmartPosException e) {
		}
	}

	@Deprecated
	private void printLogo() {
		//NOTE: we recommend skip bitmap print if always used battery for power supply.because low battery voltage maybe cause bitmap print failed.
		try {
			//set heating parameter for black white logo
			mPrinterManager.cmdSetHeatingParam(7, 100, 10);
			//usually,print logo at first.you could choose print logo form NVRAM or directly
			//for serialport printer we recommend print logo from NVRAM for good print quality
			if (mPrinterManager.getPrinterType() == PrinterManager.PRINTER_TYPE_SERIAL) {
				if (Utils.getNvramFlag(mContext)) {
					printLogoFromNVRAM();
				}
			}

			//print logo
			BitmapFactory.Options options = new BitmapFactory.Options();
			options.inTargetDensity = 0;
			options.inScaled = false;//keep original size
			Bitmap tmp = BitmapFactory.decodeResource(mContext.getResources(), R.drawable.logo_large, options);
			//Bitmap logo = ImageUtils.bmpToGrayDithering(tmp, 240, false, ImageUtils.DitherType.ATKINSON); 
			Bitmap logo = ImageUtils.bmpToBlackWhite(tmp, 215, true);//black white inverse 
			final int totalDots = mPrinterManager.getDotsPerLine();
			int xPos = (totalDots - logo.getWidth())>>1;//horizontal center
			int yPos = 0;
			//the more black area of bitmap, the more delay(slower speed)
			mPrinterManager.cmdBitmapPrint(logo, xPos, 0);
			//mPrinterManager.cmdBitmapPrintEx(logo, xPos, yPos);
			mPrinterManager.cmdLineFeed();
			tmp.recycle();
			logo.recycle();
			tmp = logo = null;

			//print logo2
			tmp = BitmapFactory.decodeResource(mContext.getResources(), R.drawable.logo, options);
			logo = ImageUtils.bmpToBlackWhite(tmp, 220, true);//black white inverse 
			xPos = (totalDots - logo.getWidth())>>1;//horizontal center
			mPrinterManager.cmdBitmapPrint(logo, xPos, 0);
			mPrinterManager.cmdLineFeed();
			tmp.recycle();
			logo.recycle();
			tmp = logo = null;

			//restore heating parameter for text
			if (mPrinterManager.isBuiltInSlow()) {
				mPrinterManager.cmdSetHeatingParam(7, 140, 2);
			} else {
				mPrinterManager.cmdSetHeatingParam(15, 100, 10);
			}
		} catch (SmartPosException e) {
		}
	}

	//NOTE:printer support QR code command after firmware v1.04
	private void printQRcode() {
		try {
			//set heating parameter for QRcode
			String qrCoderString = "https://www.pay-device.com";
			mPrinterManager.cmdQrCodePrint(8, PrinterManager.QR_ECC_LEVEL_M, qrCoderString);
			mPrinterManager.cmdLineFeed();
		} catch (SmartPosException e) {
		}
	}

	private void printBarcode() {
		String barCodeString = "ABC1234567890";
		try {
			mPrinterManager.cmdSetBarCodeStringPosition(PrinterManager.CODEBAR_STRING_MODE_BELOW);
			mPrinterManager.cmdSetBarCodeStringSize(PrinterManager.CODEBAR_STRING_FONT_A);
			mPrinterManager.cmdSetBarCodeWidth(2);
			mPrinterManager.cmdSetBarCodeHeight(80);
			//mPrinterManager.cmdSetBarCodeLeftSpacing(10);
			mPrinterManager.cmdBarCodePrint(PrinterManager.CODE93, barCodeString);

			// CODE128 special characters, and FNC1–4
			//  symbol	ASCII	Hex
			//  SHIFT	{S		7B,53
			//  CODEA	{A  	7B,41
			//  CODEB	{B  	7B,42
			//  CODEC	{C  	7B,43
			//  FNC1	{1  	7B,31
			//  FNC2	{2  	7B,32
			//  FNC3	{3  	7B,33
			//  FNC4	{4  	7B,34
			//  {		{{  	7B,7B

			//NOTE: if not specify 128A/B/C then default is auto mode.
			//barCodeString = "ABCDEFGHIJKLMNOPQRSTUVWXYZ1234";	//example: auto mode
			//barCodeString = "{AABC{Babc123";					//example: 128A+128B, 128A encode 'ABC' and 128B encode 'abc123'
			//barCodeString = "{BABCDEFGHIJKLMNOPQRSTUVWXYZ12";	//example: 128B encode
			//barCodeString = "112233";							//example: 128C encode, only support encode two digit and also string length is even
			//NOTE:long barcode need set width to 1,also need adjust heating parameter
			//if (isLongBarcode(barCodeString)) {
			//	if (mPrinterManager.getPrinterType() == PrinterManager.PRINTER_TYPE_SERIAL) {
			//		mPrinterManager.cmdSetHeatingParam(9, 140, 10);
			//	}
			//	mPrinterManager.cmdSetBarCodeWidth(1);
			//} else {
			//	mPrinterManager.cmdSetBarCodeWidth(2);
			//}
			//mPrinterManager.cmdSetBarCodeWidth(2);
			//mPrinterManager.cmdBarCodePrint(PrinterManager.CODE128, string2Code128CBytes(barCodeString));

			mPrinterManager.cmdLineFeed();
		} catch (SmartPosException e) {
		}
	}

	/**
	 * @brief convert string to 128C two digits byte array
	 *
	 * @param str barcode human readable interpretation(HRI) string
	 *
	 * @return encodes two digits byte array
	 */
	private byte[] string2Code128CBytes(String str) {
		int i;
		int len = str.length()>>1;
		byte[] result = new byte[len+2];
		result[0] = (byte)0x7B;//{C
		result[1] = (byte)0x43;
		for(i=0;i<len;i++) {
			String digit = str.substring(i<<1, ((i<<1)+2));
			result[i+2] = (byte)Integer.parseInt(digit, 10);
		}
		return result;
	}

	private boolean isLongBarcode(String data) {
		int spec = 0;
		int len = data.length();
		char[] array = data.toCharArray();
		for(int i=0;i<len;i++) {
			if (array[i] == '{') {
				spec+=2;
				i+=1;
			}
		}
		Log.d(TAG, "barcode len:"+(len-spec));
		return ((len-spec)>14)?(true):(false);
	}

	/**
	* @brief Get character count for real print, default one character is 12 dots
	*
	* @param str string your want print
	*
	* @return real print count
	*/
	private int getCount(String str) {
		int len = 0;
		//NOTE: this demo just consider CJK characters are 24 print dots, pls check your language is 12 or 24 dots 
		String range = "[\u4E00-\u9FBB]";
		for (int i=0; i<str.length(); i++) {
			String tmp = str.substring(i, i+1);
			if (tmp.matches(range)) {
				len += 2;
			} else {
				len += 1;
			}
		}
		return len;
	}

	/**
	* @brief Return blank string
	*
	* @param len blank count
	*
	* @return blank string
	*/
    private String getBlank(int len) {
        String str = "";
        for (int i=0; i<len; i++) {
            str += " ";
        }
        return str;
    }

	public boolean validate() {
		if (TextUtils.isEmpty(merchantName)
				||TextUtils.isEmpty(merchantNo)
				||TextUtils.isEmpty(terminalNo)
				||TextUtils.isEmpty(operatorNo)
				||TextUtils.isEmpty(issuer)
				||TextUtils.isEmpty(cardNo)
				||TextUtils.isEmpty(txnType)
				||TextUtils.isEmpty(batchNo)
				||TextUtils.isEmpty(voucherNo)
				||TextUtils.isEmpty(authNo)
				||TextUtils.isEmpty(expDate)
				||TextUtils.isEmpty(refNo)
				||TextUtils.isEmpty(date)
				||TextUtils.isEmpty(amount)){
			return false;
		}
		return true;
	}

	public void setMerchantName(String str) {
		this.merchantName = str;
	}

	public String getMerchantName() {
		return merchantName;
	}

	public void setMerchantNo(String str) {
		this.merchantNo = str;
	}

	public String getMerchantNo() {
		return merchantNo;
	}

	public void setOperatorNo(String str) {
		this.operatorNo = str;
	}

	public String getOperatorNo() {
		return operatorNo;
	}

	public void setTerminalNo(String str) {
		this.terminalNo = str;
	}

	public String getTerminalNo() {
		return terminalNo;
	}

	public void setIssuer(String str) {
		this.issuer = str;
	}

	public String getIssuer() {
		return issuer;
	}

	public void setAcquirer(String str) {
		this.acquirer = str;
	}

	public String getAcquirer() {
		return acquirer;
	}

	public void setCardNo(String str) {
		this.cardNo = str;
	}

	public String getCardNo() {
		return cardNo;
	}

	public void setTxnType(String str) {
		this.txnType = str;
	}

	public String getTxnType() {
		return txnType;
	}

	public void setBatchNo(String str) {
		this.batchNo = str;
	}

	public String getBatchNo() {
		return batchNo;
	}

	public void setVoucherNo(String str) {
		this.voucherNo = str;
	}

	public String getVoucherNo() {
		return voucherNo;
	}

	public void setAuthNo(String str) {
		this.authNo = str;
	}

	public String getAuthNo() {
		return authNo;
	}

	public void setExpDate(String str) {
		this.expDate = str;
	}

	public String getExpDate() {
		return expDate;
	}

	public void setRefNo(String str) {
		this.refNo = str;
	}

	public String getRefNo() {
		return refNo;
	}

	public void setDate(String str) {
		this.date = str;
	}

	public String getDate() {
		return date;
	}

	public void setAmount(String str) {
		this.amount = str;
	}

	public String getAmount() {
		return amount;
	}

	public void setTotal(String str) {
		this.total = str;
	}

	public String getTotal() {
		return total;
	}

	public void setReference(String str) {
		this.reference = str;
	}

	public String getReference() {
		return reference;
	}

	public void setPhone(String str) {
		this.phone = str;
	}

	public String getPhone() {
		return phone;
	}

	public void setTemplateId(int id) { mTemplateId = id; }
	public int getTemplateId() { return mTemplateId; }
}
