package com.paydevice.websocketserver.proxy;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Typeface;
import android.text.Layout;
import android.text.StaticLayout;
import android.text.TextDirectionHeuristics;
import android.text.TextPaint;
import android.util.Log;
import android.hardware.usb.UsbDevice;

import com.paydevice.smartpos.sdk.SmartPosException;
import com.paydevice.smartpos.sdk.printer.Printer;
import com.paydevice.smartpos.sdk.printer.PrinterManager;
import com.paydevice.smartpos.sdk.printer.SerialPortPrinter;
import com.paydevice.smartpos.sdk.printer.UsbPrinter;

import java.util.List;

public class PrinterProxy {
	private final static String TAG = "PrinterProxy";

	private Context mContext;
	private Printer mPrinter = null;
	private PrinterManager mPrinterManager = null;

	public boolean connectBuiltinPrinter(Context context, int paperWidth) {
		if (mPrinter == null) {
			//if usb printer no found then try serialport printer 
			try {
				//80mm USB printer
				mPrinter = new UsbPrinter(context);
				mPrinter.selectBuiltInPrinter();
				mPrinter.open();
				mPrinter.close();
			} catch (SmartPosException e) {
				Log.d(TAG,"no usb printer,try serialport printer");
				//58mm serialport printer
				mPrinter = new SerialPortPrinter();
				mPrinter.selectBuiltInPrinter();
			}

			mPrinterManager = new PrinterManager(mPrinter,
					(paperWidth == 80)
					? PrinterManager.TYPE_PAPER_WIDTH_80MM
					: PrinterManager.TYPE_PAPER_WIDTH_58MM);
		}
		try {
			mPrinterManager.connect();
		} catch (SmartPosException e) {
			Log.d(TAG,"connect failed");
            return false;
		}
        return true;
	}

	public boolean connectSerialPrinter(Context context, String port, int baudrate, int paperWidth) {
		if (mPrinter == null) {
            mPrinter = new SerialPortPrinter();
            ((SerialPortPrinter)mPrinter).selectPrinter(port, baudrate);
			mPrinterManager = new PrinterManager(mPrinter,
					(paperWidth == 80)
					? PrinterManager.TYPE_PAPER_WIDTH_80MM
					: PrinterManager.TYPE_PAPER_WIDTH_58MM);
		}
		try {
			mPrinterManager.connect();
		} catch (SmartPosException e) {
			Log.d(TAG,"connect failed");
            return false;
		}
        return true;
	}

	public boolean connectUsbPrinter(Context context, int vid, int pid, int paperWidth) {
		if (mPrinter == null) {
            boolean found = false;
            mPrinter = new UsbPrinter(context);
            List<UsbDevice> list = ((UsbPrinter)mPrinter).getPrinterList();
            if (list == null) {
                return false;
            }
            for (UsbDevice device: list) {
                if (device.getVendorId() == vid && device.getProductId() == pid) {
					((UsbPrinter)mPrinter).selectPrinter(device);
                    found = true;
					break;					
				}
            }
            if (!found) {
                return false;
            }
			mPrinterManager = new PrinterManager(mPrinter,
					(paperWidth == 80)
					? PrinterManager.TYPE_PAPER_WIDTH_80MM
					: PrinterManager.TYPE_PAPER_WIDTH_58MM);
		}
		try {
			mPrinterManager.connect();
		} catch (SmartPosException e) {
			Log.d(TAG,"connect failed");
            return false;
		}
        return true;
	}

	public void disconnect() {
		try {
			mPrinterManager.disconnect();
            mPrinter = null;
		} catch (SmartPosException e) {
			Log.d(TAG,"disconnect failed");
		}
	}

	public String getPrinterType() {
		String type = "serial";
		if(mPrinterManager.getPrinterType() == PrinterManager.PRINTER_TYPE_USB) {
			type = "usb";
		}
		return type;
	}

	public boolean checkPaper() {
		try {
			mPrinterManager.checkPaper();
			return true;
		} catch (SmartPosException e) {
			Log.d(TAG,"no paper");
		}
		return false;
	}

	public void setStringEncoding(String encoding) {
		try {
			mPrinterManager.setStringEncoding(encoding);
		} catch (SmartPosException e) {
			Log.d(TAG,"setStringEncoding fail");
		}
	}

	public void sendData(String data) {
		try {
			mPrinterManager.sendData(data);
		} catch (SmartPosException e) {
			Log.d(TAG,"sendData fail");
		}
	}
	public void sendData(String data, String encoding) {
		try {
			mPrinterManager.sendData(data, encoding);
		} catch (SmartPosException e) {
			Log.d(TAG,"sendData fail");
		}
	}
	public void cmdLineFeed() {
		try {
			mPrinterManager.cmdLineFeed();
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdLineFeed fail");
		}
	}
	public void cmdLineFeed(int n) {
		try {
			mPrinterManager.cmdLineFeed(n);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdLineFeed fail");
		}
	}
	public void cmdJumpTab() {
		try {
			mPrinterManager.cmdJumpTab();
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdJumpTab fail");
		}
	}
	public void cmdUnSetTable() {
		try {
			mPrinterManager.cmdUnSetTable();
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdUnSetTable fail");
		}
	}
	public void cmdSetTable(byte[] offsets) {
		try {
			mPrinterManager.cmdSetTable(offsets);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetTable fail");
		}
	}
	public void cmdSetDefaultLineSpacing() {
		try {
			mPrinterManager.cmdSetDefaultLineSpacing();
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetDefaultLineSpacing fail");
		}
	}
	public void cmdSetLineSpacing(int dots) {
		try {
			mPrinterManager.cmdSetLineSpacing(dots);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetLineSpacing fail");
		}
	}
	public void cmdSetAlignMode(int mode) {
		try {
			mPrinterManager.cmdSetAlignMode(mode);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetAlignMode fail");
		}
	}
	public void cmdSetPrintOffset(int offset) {
		try {
			mPrinterManager.cmdSetPrintOffset(offset);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetPrintOffset fail");
		}
	}
	public void cmdSetPrintMode(int mode) {
		try {
			mPrinterManager.cmdSetPrintMode(mode);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetPrintMode fail");
		}
	}
	public void cmdSetUnderlineHeight(int n) {
		try {
			mPrinterManager.cmdSetUnderlineHeight(n);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetUnderlineHeight fail");
		}
	}
	public void cmdSetFontScaleSize(int scaleWidth, int scaleHeight) {
		try {
			mPrinterManager.cmdSetFontScaleSize(scaleWidth,scaleHeight);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetFontScaleSize fail");
		}
	}
	public void cmdSetBarCodeStringPosition(int mode) {
		try {
			mPrinterManager.cmdSetBarCodeStringPosition(mode);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetBarCodeStringPosition fail");
		}
	}
	public void cmdSetBarCodeStringSize(int size) {
		try {
			mPrinterManager.cmdSetBarCodeStringSize(size);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetBarCodeStringSize fail");
		}
	}
	public void cmdSetBarCodeHeight(int n) {
		try {
			mPrinterManager.cmdSetBarCodeHeight(n);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetBarCodeHeight fail");
		}
	}
	public void cmdSetBarCodeWidth(int n) {
		try {
			mPrinterManager.cmdSetBarCodeWidth(n);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetBarCodeWidth fail");
		}
	}
	public void cmdSetBarCodeLeftSpacing(int n) {
		try {
			mPrinterManager.cmdSetBarCodeLeftSpacing(n);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetBarCodeLeftSpacing fail");
		}
	}
	public void cmdBarCodePrint(int type, String string) {
		try {
			mPrinterManager.cmdBarCodePrint(type, string);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdBarCodePrint fail");
		}
	}
	public void cmdBitmapPrint(Bitmap bitmap, int left, int top) {
		try {
			mPrinterManager.cmdBitmapPrint(bitmap, left, top);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdBitmapPrint fail");
		}
	}
	public void cmdBitmapPrintEx(Bitmap bitmap, int left, int top) {
		try {
			mPrinterManager.cmdBitmapPrintEx(bitmap, left, top);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdBitmapPrintEx fail");
		}
	}
	public void cmdSetHeatingParam(int dots, int time, int interval) {
		try {
			mPrinterManager.cmdSetHeatingParam(dots, time, interval);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetHeatingParam fail");
		}
	}
	public void cmdSetPrintDensity(int density, int delay) {
		try {
			mPrinterManager.cmdSetPrintDensity(delay, delay);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetPrintDensity fail");
		}
	}
	public void cmdSetCharacterSet(int set) {
		try {
			mPrinterManager.cmdSetCharacterSet(set);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetCharacterSet fail");
		}
	}
	public void cmdSetPrinterLanguage(int code) {
		try {
			mPrinterManager.cmdSetPrinterLanguage(code);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdSetPrinterLanguage fail");
		}
	}
	public void cmdQrCodePrint(int version, int ecc, String data) {
		try {
			mPrinterManager.cmdQrCodePrint(version, ecc, data);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdQrCodePrint fail");
		}
	}
	public void cmdPrintMultipleLines(int lineCount, int[] lineStartPos, int[] lineEndPos) {
		try {
			mPrinterManager.cmdPrintMultipleLines(lineCount, lineStartPos, lineEndPos);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdPrintMultipleLines fail");
		}
	}

	public void cmdPrintTest() {
		try {
			mPrinterManager.cmdPrintTest();
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdPrintTest fail");
		}
	}
	public void cmdCutPaper(int mode) {
		try {
			mPrinterManager.cmdCutPaper(mode);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdCutPaper fail");
		}
	}

	public void cmdPrintBitmapFromNVRAM(int index, int zoom) {
		try {
			mPrinterManager.cmdPrintBitmapFromNVRAM(index, zoom);
		} catch (SmartPosException e) {
			Log.d(TAG,"cmdPrintBitmapFromNVRAM fail");
		}
	}
	public boolean isBuiltInSlow() {
		return mPrinterManager.isBuiltInSlow();
	}

	public void printDevidingLine() {
		final int totalDots = mPrinterManager.getDotsPerLine();
		int[] pLineStartPosition_58 = {0,192};
		int[] pLineEndPosition_58 = {192,383};
		int[] pLineStartPosition_80 = {0,200,400};
		int[] pLineEndPosition_80 = {200,400,575};
		int height=5;
		while(height-- > 0) {
			try {
				if (totalDots == 384)
					mPrinterManager.cmdPrintMultipleLines(pLineStartPosition_58.length, pLineStartPosition_58, pLineEndPosition_58);
				else
					mPrinterManager.cmdPrintMultipleLines(pLineStartPosition_80.length, pLineStartPosition_80, pLineEndPosition_80);
			} catch (SmartPosException e) {
			}
		}
		try {
			mPrinterManager.cmdLineFeed(1);
		} catch (SmartPosException e) {
		}
	}

    public boolean saveBitmaptoNVRAM(Bitmap[] bmps) {
        boolean ret = true;
        try {
            mPrinterManager.connect();
            mPrinterManager.cmdSaveBitmapToNVRAM(bmps);
        } catch (SmartPosException e) {
            Log.d(TAG, "save bitmap to NVRAM err:" + e.getErrorCode());
            ret = false;
        } finally {
            try {
                Thread.sleep(500);
            } catch (InterruptedException ie) {
            }
			try {
				mPrinterManager.disconnect();
			} catch (SmartPosException e) {
			}
        }
        return ret;
    }

    public boolean deleteBitmapFromNVRAM() {
        boolean ret = true;
        try {
            mPrinterManager.connect();
            mPrinterManager.cmdDeleteBitmapFromNVRAM();
        } catch (SmartPosException e) {
            Log.d(TAG, "delete bitmaps from NVRAM err:" + e.getErrorCode());
            ret = false;
        } finally {
            try {
                Thread.sleep(500);
            } catch (InterruptedException ie) {
            }
			try {
				mPrinterManager.disconnect();
			} catch (SmartPosException e) {
			}
        }
        return ret;
	}

    /**
     * @brief Print text and feed one line by printer built-in codepage
     *
     * @param data print text
     *
     * @return 
     */
	public void printLine(String data) throws SmartPosException {
		mPrinterManager.sendData(data);
		mPrinterManager.cmdLineFeed();
	}

	/**
	* @brief Print one line text table with 2 columns by printer built-in codepage, left text align left and right text align right
	*
	* @param leftText left text
	* @param rightText right text
	*
	* @return 
	*/
	public void printLine(String leftText, String rightText) throws SmartPosException {
		String[] arrStr = new String[] {leftText, rightText};
		int[] arrWidth = new int[] {1, 2};//ratio 1:2
		int[] arrAlign = new int[] {PrinterManager.ALIGN_LEFT, PrinterManager.ALIGN_RIGHT};
		printLine(arrStr, arrWidth, arrAlign);
	}

    /**
     * @brief Print one line text table with multi columns by printer built-in codepage, implemented with blank
     *
     * @param arrStr items array
     * @param arrWidth items width array
     * @param arrAlign items align mode array
     *
     * @return 
     */
    public void printLine(String[] arrStr, int[] arrWidth, int[] arrAlign) throws SmartPosException {
        String data = ""; 
		final int totalDots = mPrinterManager.getDotsPerLine();
        int totalRatio = 0;
        for(int ratio : arrWidth) {
            totalRatio += ratio;
        }
        for(int i=0; i<arrStr.length; i++) {
            int width = Math.round((totalDots/12) * (arrWidth[i] / (float)totalRatio));
            String item = (getCount(arrStr[i]) > width) ? (arrStr[i].substring(0, width)) : (arrStr[i]);//cut string if too long
            if(arrAlign[i] == PrinterManager.ALIGN_LEFT){
                data += (item + getBlank(width - getCount(item)));
            } else if (arrAlign[i] == PrinterManager.ALIGN_MIDDLE) {
                String leftBlank = getBlank((width - getCount(item)) / 2);
                String rightBlank = leftBlank;
                int rightLen = width - getCount(item) - leftBlank.length() * 2;
                if(rightLen > 0) { 
                    rightBlank += getBlank(rightLen);
                }
                data += leftBlank + item + rightBlank;
            } else if(arrAlign[i] == PrinterManager.ALIGN_RIGHT){
                data += (getBlank(width - getCount(item)) + item);
            }
        }
		mPrinterManager.sendData(data);
		mPrinterManager.cmdLineFeed();
	}

	/**
	* @brief Print one line text by bitmap print.
	*
	* @param data
	*
	* @return 
	*/
	public void printLineByBitmap(String data) throws SmartPosException {
		String[] arrStr = new String[] {data};
		int[] arrWidth = new int[] {100};//ratio 100%
		int[] arrAlign = new int[] {PrinterManager.ALIGN_LEFT};
		printLineByBitmap(arrStr, arrWidth, arrAlign);
	}

	/**
	* @brief Print one line text table with 2 columns by bitmap print, left text align left and right text align right
	*
	* @param leftText left text
	* @param rightText right text
	*
	* @return 
	*/
	public void printLineByBitmap(String leftText, String rightText) throws SmartPosException {
		String[] arrStr = new String[] {leftText, rightText};
		int[] arrWidth = new int[] {1, 2};//ratio 1:2
		int[] arrAlign = new int[] {PrinterManager.ALIGN_LEFT, PrinterManager.ALIGN_RIGHT};
		printLineByBitmap(arrStr, arrWidth, arrAlign);
	}

	/**
	* @brief Print one line text table with multi columns by bitmap print. 
	*
    * @param arrStr items array
    * @param arrWidth items width array
    * @param arrAlign items align mode array
	*
	* @return 
	*/
	public void printLineByBitmap(String[] arrStr, int[] arrWidth, int[] arrAlign) throws SmartPosException {
		final float size = 24f;//default text size is 24px(24dots)
		final float lineSpacingAdd = 4f;
		final int lineWidth = mPrinterManager.getDotsPerLine();
		if (arrStr.length > 3) {
			Log.w(TAG,"string too long, maybe the string cannot printed completely");
		}
		//setup text paint
		TextPaint textPaint = new TextPaint();
		textPaint.setColor(Color.BLACK);
		textPaint.setStyle(Paint.Style.FILL_AND_STROKE);
		textPaint.setStrokeCap(Paint.Cap.SQUARE);
		textPaint.setStrokeWidth(0);
		textPaint.setTextSize(size);
		//android system font include most languages of world, but you can also used Typefase to load external .ttf/.otf font.
		Typeface font = Typeface.create(Typeface.SANS_SERIF, Typeface.NORMAL);
		textPaint.setTypeface(font);
		//metrics font
		Paint.FontMetrics fontMetrics = textPaint.getFontMetrics();
		final float yPos = fontMetrics.descent-fontMetrics.ascent-size;
		final int lineHeight = Math.round(fontMetrics.descent-fontMetrics.ascent+yPos+lineSpacingAdd);

		int totalRatio = 0;
		for(int ratio : arrWidth) {
			totalRatio += ratio;
		}
		//use a bitmap to load one line string, bitmap width is always lineWidth(384px for 58mm).
		//The part that exceeds the width will not be print
		Bitmap bmp = Bitmap.createBitmap(lineWidth, lineHeight, Bitmap.Config.ARGB_8888);
		Canvas canvas = new Canvas(bmp);
		canvas.drawColor(Color.WHITE);

		int previousStrGraphWidth = 0;
		//draw each text part to bitmap
		for(int i=0; i<arrStr.length; i++) {
			int strGraphWidth = arrWidth[i]*lineWidth/totalRatio;
			if (strGraphWidth < 12) {
				Log.e(TAG,"width too small,check the ratio");
				continue;
			}
			StaticLayout.Builder builder = StaticLayout.Builder.obtain(arrStr[i], 0, arrStr[i].length(), textPaint, strGraphWidth)
				.setTextDirection(TextDirectionHeuristics.LTR)//Always decides that the direction is left to right.
				.setLineSpacing(lineSpacingAdd, 1.0f)//line spacing mult 1.0
				.setIncludePad(false);
			//set align mode for each string
			switch(arrAlign[i]) {
				case PrinterManager.ALIGN_LEFT:
					builder.setAlignment(Layout.Alignment.ALIGN_NORMAL);
					break;
				case PrinterManager.ALIGN_MIDDLE:
					builder.setAlignment(Layout.Alignment.ALIGN_CENTER);
					break;
				case PrinterManager.ALIGN_RIGHT:
					builder.setAlignment(Layout.Alignment.ALIGN_OPPOSITE);
					break;
			}
			StaticLayout layout = builder.build();
			//Log.d(TAG,"lines:"+layout.getLineCount()+" lineHeight:"+lineHeight+" strWidth:"+strGraphWidth);
			//Log.d(TAG,"top:"+fontMetrics.top+" leading:"+fontMetrics.leading+" ascent:"+fontMetrics.ascent
			//		+" yPos:"+yPos+" descent:"+fontMetrics.descent+" bottom:"+fontMetrics.bottom);
			canvas.translate(previousStrGraphWidth, yPos);
			layout.draw(canvas);
			previousStrGraphWidth += strGraphWidth;
		}
		mPrinterManager.cmdBitmapPrint(bmp, 0, 0);
		//try used cmdSetLineSpacing and cmdLineFeed(n) if you want setup line spacing for printLineByBitmap

		bmp.recycle();
		bmp = null;
	}

	public void printMulitLineByBitmap(String text) throws SmartPosException {
		final float lineSpacingAdd = 4f;
		final int bmpWidth = mPrinterManager.getDotsPerLine();
		TextPaint textPaint = new TextPaint();
		Typeface font = Typeface.create(Typeface.SANS_SERIF, Typeface.NORMAL);
		textPaint.setColor(Color.BLACK); 
		textPaint.setStyle(Paint.Style.FILL_AND_STROKE);
		textPaint.setStrokeCap(Paint.Cap.SQUARE);
		textPaint.setStrokeWidth(0);
		textPaint.setTextSize(24); 
		textPaint.setTypeface(font);
		Paint.FontMetrics fontMetrics = textPaint.getFontMetrics();
		final float yPos = fontMetrics.descent-fontMetrics.ascent-24;

		mPrinterManager.checkPaper();
		String[] lines = text.split("\\n");
		for (String str: lines) {
			StaticLayout.Builder builder = StaticLayout.Builder.obtain(str, 0, str.length(), textPaint, bmpWidth)
				.setLineSpacing(lineSpacingAdd, 1.0f)//line spacing mult 1.0
				.setIncludePad(false);
			StaticLayout layout = builder.build();
			final int bmpHeight = Math.round(layout.getLineCount()*(fontMetrics.descent-fontMetrics.ascent+lineSpacingAdd)+yPos);
			Bitmap bmp = Bitmap.createBitmap(bmpWidth, bmpHeight, Bitmap.Config.ARGB_8888);
			Canvas canvas = new Canvas(bmp);
			canvas.drawColor(Color.WHITE);
			canvas.translate(0, yPos);
			layout.draw(canvas);
			Log.d(TAG,"lines:"+layout.getLineCount()+" bmpHeight:"+bmpHeight);
			Log.d(TAG,"top:"+fontMetrics.top+" leading:"+fontMetrics.leading+" ascent:"+fontMetrics.ascent
					+" yPos:"+yPos+" descent:"+fontMetrics.descent+" bottom:"+fontMetrics.bottom);
			mPrinterManager.cmdBitmapPrint(bmp, 0, 0);
			bmp.recycle();
			bmp = null;
		}
	}

	private boolean isLongBarcode(String data) {
		int spec = 0;
		int len = data.length();
		char[] array = data.toCharArray();
		for(int i=0;i<len;i++) {
			if (array[i] == '{') {
				spec+=2;
				i+=1;
			}
		}
		Log.d(TAG, "barcode len:"+(len-spec));
		return ((len-spec)>14)?(true):(false);
	}

	/**
	* @brief Get character count for real print, default one character is 12 dots
	*
	* @param str string your want print
	*
	* @return real print count
	*/
	private int getCount(String str) {
		int len = 0;
		//NOTE: this demo just consider CJK characters are 24 print dots, pls check your language is 12 or 24 dots 
		String range = "[\u4E00-\u9FBB]";
		for (int i=0; i<str.length(); i++) {
			String tmp = str.substring(i, i+1);
			if (tmp.matches(range)) {
				len += 2;
			} else {
				len += 1;
			}
		}
		return len;
	}

	/**
	* @brief Return blank string
	*
	* @param len blank count
	*
	* @return blank string
	*/
    private String getBlank(int len) {
        String str = "";
        for (int i=0; i<len; i++) {
            str += " ";
        }
        return str;
    }


}
