let timeout;

// 显示通知
function showNotification(title, message, duration = 5000) {
    clearTimeout(timeout);
    const existingNotification = document.querySelector('.notification-modal');
    if (existingNotification) {
        existingNotification.remove();
    }

    const modal = document.createElement('div');
    modal.className = 'notification-modal';
    
    const modalContent = document.createElement('div');
    modalContent.className = 'notification-content';

    const titleElement = document.createElement('div');
    titleElement.className = 'modal-title';
    titleElement.textContent = title;

    const closeButton = document.createElement('button');
    closeButton.className = 'modal-close';
    closeButton.innerHTML = '×';
    closeButton.onclick = () => modal.remove();

    const messageElement = document.createElement('div');
    messageElement.className = 'modal-message';
    messageElement.textContent = message;

    modalContent.appendChild(titleElement);
    modalContent.appendChild(closeButton);
    modalContent.appendChild(messageElement);
    modal.appendChild(modalContent);
    document.body.appendChild(modal);

    // 在指定时间后自动关闭
    timeout = setTimeout(() => {
        modal.remove();
    }, duration);
}

// 生成示例餐厅账单
function generateSampleBill() {
    return `
        RESTAURANT BILL
        ===============
        Date: ${new Date().toLocaleString()}
        
        Items:
        1. Burger      $10.99
        2. Fries       $3.99
        3. Soda        $1.99
        
        Subtotal:      $16.97
        Tax (8%):      $1.36
        ---------------
        Total:         $18.33
        
        Thank you for dining with us!
    `;
}

// 测试 USB 打印机
async function testUsbPrinter() {
    try {
        // list usb printer by VID
        const device = await navigator.usb.requestDevice({ filters: [ {vendorId: 0x4648} ] });
        console.log('found:', device.productName);
        
        await device.open();
        await device.selectConfiguration(1);
        await device.claimInterface(0);
        
        const config = device.configuration;
        const iface = config.interfaces[0];
        const endpoints = iface.alternate.endpoints;
        
        console.log('endpoints list:');
        endpoints.forEach(ep => {
            console.log(`  endpoint ${ep.endpointNumber}: ${ep.direction}, ${ep.type}`);
        });
        
        let outEndpoint = endpoints.find(ep => ep.direction === 'out');
        if (!outEndpoint) outEndpoint = endpoints[0];
        
        const sampleBill = generateSampleBill();
        const textEncoder = new TextEncoder();
        const billData = textEncoder.encode(sampleBill);

        const leftAlign = [0x1B, 0x61, 0x00]; // ESC a 0
        const lineFeed = [0x0A];
        const cutCommand = [0x1B, 0x6D]; // ESC m (half cut)
        const printData = [
            ...leftAlign,
            ...billData,
            ...lineFeed,
            ...lineFeed,
            ...lineFeed,
            ...lineFeed,
            ...lineFeed,
            ...cutCommand
        ];

        const data = new Uint8Array(printData);
        await device.transferOut(outEndpoint.endpointNumber, data.buffer);

        console.log('print done');
        showNotification('success', 'print done');
        
    } catch (error) {
        console.error('USB Printer Error:', error);
        if (error.name === 'NotFoundError') {
            showNotification('USB Printer Error', 'No compatible USB printer found. Please make sure your printer is connected and try again.');
        } else if (error.name === 'SecurityError') {
            showNotification('USB Printer Error', 'Access to the USB device was denied. Please make sure you have the necessary permissions and try again.');
        } else {
            showNotification('USB Printer Error', `Failed to connect to the USB printer: ${error.message}. Please check your connection and try again.`);
        }
    } finally {
        if (device && device.opened) {
            try {
                await device.close();
            } catch (closeError) {
                console.error('关闭 USB 设备时出错:', closeError);
            }
        }
    }
}

// 测试蓝牙打印机
async function testBluetoothPrinter() {
	let device = null;
	let server = null;

	try {
			const serviceUUIDs = [
			'0000ae00-0000-1000-8000-00805f9b34fb', // 一些打印机的服务 UUID
			'00001101-0000-1000-8000-00805f9b34fb', // SPP 串口服务
			];

		// 检查浏览器支持
		if (!('bluetooth' in navigator)) {
			showNotification('Bluetooth Printer Error', 'Bluetooth is not supported in this browser. Please use Chrome or Edge.');
			return;
		}

		// 请求连接蓝牙设备
		device = await navigator.bluetooth.requestDevice({
			acceptAllDevices: true,  // 接受所有设备
			optionalServices: serviceUUIDs
		});

		console.log(`找到设备: ${device.name || '未知设备'}, ID: ${device.id}`);

		// 连接到设备
		server = await device.gatt.connect();
		console.log('已连接到设备');

		// 获取所有服务
		const services = await server.getPrimaryServices();
		console.log(`找到 ${services.length} 个服务:`);

		services.forEach(service => {
				console.log(`- 服务 UUID: ${service.uuid}`);
				});

		// 尝试查找可用的服务
		let selectedService = null;
		for (const serviceUUID of serviceUUIDs) {
			try {
				selectedService = await server.getPrimaryService(serviceUUID);
				console.log(`成功获取服务: ${serviceUUID}`);
				break;
			} catch (e) {
				// 继续尝试下一个服务
				continue;
			}
		}

		if (!selectedService) {
			// 如果没有匹配的标准服务，使用第一个可用的服务
			selectedService = services[0];
			console.log(`使用第一个可用服务: ${selectedService.uuid}`);
		}

		// 获取服务的所有特征
		const characteristics = await selectedService.getCharacteristics();
		console.log(`找到 ${characteristics.length} 个特征:`);

		characteristics.forEach(char => {
				console.log(`- 特征 UUID: ${char.uuid}, 属性:`, {
				read: char.properties.read,
				write: char.properties.write,
				notify: char.properties.notify,
				indicate: char.properties.indicate
				});
		});

		// 查找可写的特征
		const writableChar = characteristics.find(c => c.properties.write);
		if (!writableChar) {
			showNotification('Bluetooth Printer', 'No writable characteristic found. Cannot send print data.');
			return;
		}

		console.log(`使用特征: ${writableChar.uuid}`);

		// 发送数据到打印机
        const sampleBill = generateSampleBill();
        const textEncoder = new TextEncoder();
        const billData = textEncoder.encode(sampleBill);

        const leftAlign = [0x1B, 0x61, 0x00]; // ESC a 0
        const lineFeed = [0x0A];
        const cutCommand = [0x1B, 0x6D]; // ESC m (half cut)
        const data = [
            ...leftAlign,
            ...billData,
            ...lineFeed,
            ...lineFeed,
            ...lineFeed,
            ...lineFeed,
            ...lineFeed,
            ...cutCommand
        ];

        const printData = new Uint8Array(data);
		console.log(`总数据大小: ${data.length} 字节`);

		const chunkSize = 197; //MTU=200
		const totalChunks = Math.ceil(printData.length / chunkSize);

		for (let i = 0; i < printData.length; i += chunkSize) {
			const chunk = printData.slice(i, i + chunkSize);
			const chunkNumber = Math.floor(i / chunkSize) + 1;

			try {
				if (writableChar.properties.writeWithoutResponse) {
					await writableChar.writeValueWithoutResponse(chunk);
				} else {
					await writableChar.writeValue(chunk);
				}

				console.log(`发送块 ${chunkNumber}/${totalChunks}: ${chunk.length} 字节`);
				// 添加延迟，避免发送过快
				await new Promise(resolve => setTimeout(resolve, 12));
			} catch (chunkError) {
				console.error(`块 ${chunkNumber} 发送失败:`, chunkError);
				throw new Error(`数据传输失败 (块 ${chunkNumber})`);
			}
		}
		showNotification('成功', '打印任务已发送到蓝牙打印机！');

	} catch (error) {
		console.error('蓝牙打印机错误详情:', error);

		if (error.name === 'NotFoundError') {
			showNotification('蓝牙打印机错误', '未找到蓝牙设备。请确保打印机已打开且可被发现。');
		} else if (error.name === 'SecurityError') {
			showNotification('蓝牙打印机错误', '蓝牙权限被拒绝。请检查浏览器权限设置。');
		} else if (error.name === 'NetworkError') {
			showNotification('蓝牙打印机错误', '连接已断开。请确保打印机在范围内。');
		} else {
			showNotification('蓝牙打印机错误', `连接失败: ${error.message}`);
		}
	} finally {
		// 清理连接
		if (device && device.gatt.connected) {
			try {
				// 等待一小段时间确保数据发送完成
				await new Promise(resolve => setTimeout(resolve, 500));
				device.gatt.disconnect();
				console.log('已断开蓝牙连接');
			} catch (disconnectError) {
				console.error('断开连接时出错:', disconnectError);
			}
		}
	}
}

async function testSerialPrinter() {
    if (!navigator.serial) {
        console.log('浏览器不支持Web Serial API');
        return false;
    }

    try {
        const port = await navigator.serial.requestPort();
        await port.open({ baudRate: 500000 });

        const encoder = new TextEncoder();
        const writer = port.writable.getWriter();
        const data = generateSampleBill();  // 示例打印数据
        await writer.write(encoder.encode(data));
        writer.releaseLock();
        await port.close();

        return true;
    } catch (error) {
        console.error('Web Serial失败:', error);
        return false;
    }
}
